(function(wp) {
    const { registerPlugin } = wp.plugins;
    const { PluginDocumentSettingPanel } = wp.editor;
    const { PanelBody, TextControl, TextareaControl, SelectControl } = wp.components;
    const { useSelect, useDispatch } = wp.data;
    const { __ } = wp.i18n;
    const { createElement } = wp.element;

    // Custom SVG icon (search heart)
    const customIcon = createElement('svg', {
        xmlns: 'http://www.w3.org/2000/svg',
        width: 20,
        height: 20,
        viewBox: '0 0 16 16',
        fill: 'currentColor'
    },
        createElement('path', { d: 'M6.5 4.482c1.664-1.673 5.825 1.254 0 5.018-5.825-3.764-1.664-6.69 0-5.018' }),
        createElement('path', { d: 'M13 6.5a6.47 6.47 0 0 1-1.258 3.844q.06.044.115.098l3.85 3.85a1 1 0 0 1-1.414 1.415l-3.85-3.85a1 1 0 0 1-.1-.115h.002A6.5 6.5 0 1 1 13 6.5M6.5 12a5.5 5.5 0 1 0 0-11 5.5 5.5 0 0 0 0 11' })
    );

    const RichWPSEO = () => {
        const { meta } = useSelect(select => ({
            meta: select('core/editor').getEditedPostAttribute('meta')
        }));

        const { editPost } = useDispatch('core/editor');

        // Donation notice dismissal
        const dismissDonation = () => {
            fetch(richwpDonationData.ajaxUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'richwp_seo_dismiss_donation',
                    nonce: richwpDonationData.nonce
                })
            }).then(() => {
                // Refresh to hide notice
                window.location.reload();
            });
        };

        // Live character count
        const currentDesc = meta._richwp_meta_desc || '';
        const charCount = currentDesc.length;
        const charColor = charCount > 160 ? 'red' : (charCount >= 150 ? 'green' : 'orange');

        const data = richwpSeoData; // Localized data from PHP

        console.log('RichWP SEO: Donation data check', typeof richwpDonationData, richwpDonationData);

        const presetBases = {
            none: '',
            organization: {
                '@context': 'https://schema.org',
                '@type': 'Organization',
                'name': data.siteName,
                'url': data.siteUrl,
                'description': data.siteDescription
            },
            webpage: {
                '@context': 'https://schema.org',
                '@type': 'WebPage',
                'url': data.pageUrl
            },
            article: {
                '@context': 'https://schema.org',
                '@type': 'Article',
                'datePublished': data.publishDate,
                'dateModified': data.modifiedDate,
                'author': {'@type': 'Person', 'name': data.authorName},
                'publisher': {
                    '@type': 'Organization',
                    'name': data.siteName
                },
                'articleSection': data.category,
                'keywords': data.keywords
            },
            newsarticle: {
                '@context': 'https://schema.org',
                '@type': 'NewsArticle',
                'datePublished': data.publishDate,
                'dateModified': data.modifiedDate,
                'author': {'@type': 'Person', 'name': data.authorName},
                'publisher': {
                    '@type': 'Organization',
                    'name': data.siteName
                },
                'articleSection': data.category,
                'keywords': data.keywords
            },
            event: {
                '@context': 'https://schema.org',
                '@type': 'Event',
                'name': data.defaultPostTitle,
                'startDate': data.publishDate,
                'endDate': data.publishDate,
                'location': {
                    '@type': 'Place',
                    'name': 'Your Location Name',
                    'address': {
                        '@type': 'PostalAddress',
                        'streetAddress': 'Your Street Address',
                        'addressLocality': 'City',
                        'addressRegion': 'Region',
                        'postalCode': 'Postal Code',
                        'addressCountry': 'DE'
                    }
                },
                'organizer': {
                    '@type': 'Organization',
                    'name': data.siteName,
                    'url': data.siteUrl
                }
            },
            recipe: {
                '@context': 'https://schema.org',
                '@type': 'Recipe',
                'name': data.defaultPostTitle,
                'author': {'@type': 'Person', 'name': data.authorName},
                'datePublished': data.publishDate,
                'prepTime': 'PT15M',
                'cookTime': 'PT30M',
                'totalTime': 'PT45M',
                'recipeYield': '4 servings',
                'recipeIngredient': [
                    'Ingredient 1',
                    'Ingredient 2',
                    'Ingredient 3'
                ],
                'recipeInstructions': [
                    {
                        '@type': 'HowToStep',
                        'text': 'Step 1: Do something'
                    },
                    {
                        '@type': 'HowToStep',
                        'text': 'Step 2: Do something else'
                    }
                ]
            },
            faqpage: {
                '@context': 'https://schema.org',
                '@type': 'FAQPage',
                'mainEntity': [
                    {
                        '@type': 'Question',
                        'name': 'Example Question?',
                        'acceptedAnswer': {
                            '@type': 'Answer',
                            'text': 'Example answer here.'
                        }
                    }
                ]
            },
            product: {
                '@context': 'https://schema.org',
                '@type': 'Product',
                'offers': {
                    '@type': 'Offer',
                    'priceCurrency': 'USD',
                    'price': '0.00',
                    'availability': 'https://schema.org/InStock',
                    'url': data.pageUrl
                }
            },
            localbusiness: {
                '@context': 'https://schema.org',
                '@type': 'LocalBusiness',
                'name': data.siteName,
                'url': data.siteUrl,
                'telephone': '+49-123-4567890',
                'address': {
                    '@type': 'PostalAddress',
                    'streetAddress': 'Your Street Address',
                    'addressLocality': 'City',
                    'addressRegion': 'Region',
                    'postalCode': 'Postal Code',
                    'addressCountry': 'DE'
                }
            }
        };

        function buildPreset(key, customTitle) {
            const base = presetBases[key];
            if (!base) return '';

            const preset = JSON.parse(JSON.stringify(base));
            const customDesc = meta._richwp_meta_desc || '';

            // Get current featured image dynamically
            const featuredMediaId = wp.data.select('core/editor').getEditedPostAttribute('featured_media');
            let featuredImageUrl = data.featuredImage; // fallback to initial data
            if (featuredMediaId) {
                const media = wp.data.select('core').getMedia(featuredMediaId);
                if (media && media.source_url) {
                    featuredImageUrl = media.source_url;
                }
            }

            if (data.logoUrl && (key === 'organization' || key === 'article' || key === 'newsarticle')) {
                const logoObj = {'@type': 'ImageObject', 'url': data.logoUrl};
                if (key === 'organization') preset.logo = logoObj;
                if (key === 'article' || key === 'newsarticle') preset.publisher.logo = logoObj;
            }

            if (featuredImageUrl && (key === 'article' || key === 'newsarticle' || key === 'product' || key === 'event' || key === 'recipe')) {
                preset.image = featuredImageUrl;
            }

            if (key === 'organization') {
                if (customDesc) {
                    preset.description = customDesc;
                }
            } else if (key === 'webpage') {
                preset.name = customTitle;
                if (customDesc) {
                    preset.description = customDesc;
                }
            } else if (key === 'article' || key === 'newsarticle') {
                preset.headline = customTitle;
                if (customDesc) {
                    preset.description = customDesc;
                }
            } else if (key === 'product') {
                preset.name = customTitle;
                if (customDesc) {
                    preset.description = customDesc;
                }
            } else if (key === 'event') {
                preset.name = customTitle;
                if (customDesc) {
                    preset.description = customDesc;
                }
            } else if (key === 'recipe') {
                preset.name = customTitle;
                if (customDesc) {
                    preset.description = customDesc;
                }
            }

            return JSON.stringify(preset, null, 2);
        }

        return createElement(
            PluginDocumentSettingPanel,
            {
                name: 'richwp-seo-panel',
                title: 'RichWP SEO',
                icon: customIcon,
                initialOpen: true
            },
            // Donation Notice
            richwpDonationData.showNotice && createElement('div', {
                style: {
                    backgroundColor: '#fff5f5',
                    border: '1px solid #dc3232',
                    borderRadius: '4px',
                    padding: '12px',
                    marginBottom: '16px',
                    position: 'relative'
                }
            },
                createElement('button', {
                    onClick: dismissDonation,
                    style: {
                        position: 'absolute',
                        top: '8px',
                        right: '8px',
                        background: 'none',
                        border: 'none',
                        fontSize: '18px',
                        cursor: 'pointer',
                        color: '#666'
                    },
                    'aria-label': 'Dismiss donation notice'
                }, '×'),
                createElement('p', {
                    style: { margin: '0 0 8px 0', fontSize: '14px', fontWeight: 'bold' }
                }, '☕ Support RichWP SEO Development'),
                createElement('p', {
                    style: { margin: '0 0 8px 0', fontSize: '13px' }
                }, 'Enjoying this free plugin? Consider supporting ongoing development and new features.'),
                createElement('a', {
                    href: 'https://richwp.com/donate',
                    target: '_blank',
                    rel: 'noopener noreferrer',
                    style: {
                        display: 'inline-block',
                        backgroundColor: '#dc3232',
                        color: 'white',
                        padding: '6px 12px',
                        textDecoration: 'none',
                        borderRadius: '4px',
                        fontSize: '13px'
                    }
                }, 'Make a Donation')
            ),
            createElement(TextControl, {
                label: 'Page Title',
                value: meta._richwp_custom_title || '',
                onChange: (value) => editPost({ meta: { _richwp_custom_title: value } }),
                placeholder: 'Default WordPress title',
                help: 'Overrides <title> tag (no site name appended).',
                __next40pxDefaultSize: true,
                __nextHasNoMarginBottom: true
            }),
            createElement(TextareaControl, {
                label: 'Meta Description',
                value: currentDesc,
                onChange: (value) => editPost({ meta: { _richwp_meta_desc: value } }),
                rows: 3,
                help: createElement('div', {},
                    createElement('span', { style: { color: charColor, fontWeight: 'bold' } }, `Characters: ${charCount}`),
                    ' (recommended 150–160)'
                ),
                __next40pxDefaultSize: true,
                __nextHasNoMarginBottom: true
            }),
            createElement(SelectControl, {
                label: 'Schema Preset',
                value: 'none',
                options: [
                    { label: 'None / Custom', value: 'none' },
                    { label: 'Organization', value: 'organization' },
                    { label: 'WebPage', value: 'webpage' },
                    { label: 'Article', value: 'article' },
                    { label: 'NewsArticle', value: 'newsarticle' },
                    { label: 'Event', value: 'event' },
                    { label: 'Recipe', value: 'recipe' },
                    { label: 'FAQPage', value: 'faqpage' },
                    { label: 'Product', value: 'product' },
                    { label: 'LocalBusiness', value: 'localbusiness' }
                ],
                onChange: (key) => {
                    if (key === 'none') return;

                    const currentTitle = meta._richwp_custom_title || data.defaultPostTitle;

                    const template = buildPreset(key, currentTitle);

                    if (meta._richwp_schema_json && meta._richwp_schema_json.trim() !== '') {
                        if (!confirm('This will replace your current schema with the selected preset. Continue?')) {
                            return;
                        }
                    }

                    editPost({ meta: { _richwp_schema_json: template } });
                },
                help: 'Loads template – changes apply on Update.',
                __next40pxDefaultSize: true,
                __nextHasNoMarginBottom: true
            }),
            createElement(TextareaControl, {
                label: 'Schema JSON-LD',
                value: meta._richwp_schema_json || '',
                onChange: (value) => editPost({ meta: { _richwp_schema_json: value } }),
                rows: 12,
                help: 'Raw JSON only (no <script> tags). Update page to apply.',
                __next40pxDefaultSize: true,
                __nextHasNoMarginBottom: true
            })
        );
    };

    registerPlugin('richwp-seo-panel', {
        render: RichWPSEO,
        icon: customIcon
    });
})(window.wp);