<?php
/**
 * Admin Class for RichWP SEO
 *
 * Handles admin-side functionality including asset enqueuing.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class RichWP_SEO_Admin {

    const DONATION_DISMISSAL_INTERVALS = [0, WEEK_IN_SECONDS, MONTH_IN_SECONDS, 3 * MONTH_IN_SECONDS, 3 * MONTH_IN_SECONDS];
    const ACTIVATION_DELAY_SECONDS = 43200; // 12 hours

    /**
     * Initialize the admin handler.
     */
    public static function init() {
        add_action('enqueue_block_editor_assets', [__CLASS__, 'enqueue_assets']);
        add_action('admin_notices', [__CLASS__, 'show_donation_notice']);
        add_action('wp_ajax_richwp_seo_dismiss_donation', [__CLASS__, 'dismiss_donation_notice']);
    }

    /**
     * Check if donation notice should be shown.
     *
     * @return bool True if notice should be shown, false otherwise.
     */
    private static function should_show_donation_notice() {
        // Check activation delay
        $activation_time = get_option('richwp_seo_activation_time', 0);
        if ($activation_time && time() < $activation_time + self::ACTIVATION_DELAY_SECONDS) {
            return false;
        }

        $dismissal = get_option('richwp_seo_donation_dismissal', ['level' => 0, 'timestamp' => 0]);
        $current_time = time();
        $show_notice = true;

        if ($dismissal['level'] > 0) {
            $hide_until = $dismissal['timestamp'] + self::DONATION_DISMISSAL_INTERVALS[$dismissal['level']];
            if ($current_time < $hide_until) {
                $show_notice = false;
            } elseif ($dismissal['level'] >= 4) {
                // Reset to show again after 3 months for permanent dismissals
                update_option('richwp_seo_donation_dismissal', ['level' => 0, 'timestamp' => 0]);
            }
        }

        return $show_notice;
    }

    /**
     * Enqueue Gutenberg editor assets.
     */
    public static function enqueue_assets() {
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->base, ['post', 'page'])) {
            return;
        }

        // Enqueue the JavaScript file
        wp_enqueue_script(
            'richwp-seo-panel',
            plugin_dir_url(dirname(__FILE__, 2) . '/richwp-seo.php') . 'assets/js/richwp-seo-panel.js',
            ['wp-plugins', 'wp-editor', 'wp-element', 'wp-components', 'wp-data', 'wp-i18n'],
            '3.4',
            true
        );

        // Prepare dynamic data for JavaScript
        $post_id = get_the_ID();
        $categories = get_the_category($post_id);
        $category = !empty($categories) ? $categories[0]->name : '';
        $tags = get_the_tags($post_id);
        $keywords = $tags ? implode(', ', wp_list_pluck($tags, 'name')) : '';

        $data = [
            'defaultPostTitle' => get_the_title($post_id),
            'siteName' => get_bloginfo('name'),
            'siteUrl' => home_url('/'),
            'siteDescription' => get_bloginfo('description'),
            'excerpt' => has_excerpt($post_id) ? wp_strip_all_tags(get_the_excerpt($post_id)) : '',
            'featuredImage' => get_the_post_thumbnail_url($post_id, 'full') ?: '',
            'publishDate' => get_the_date('c', $post_id),
            'modifiedDate' => get_the_modified_date('c', $post_id),
            'authorName' => get_the_author_meta('display_name', get_post()->post_author),
            'pageUrl' => get_permalink($post_id),
            'category' => $category,
            'keywords' => $keywords,
            'logoUrl' => ''
        ];

        // Get site logo
        $custom_logo_id = get_theme_mod('custom_logo');
        if ($custom_logo_id) {
            $data['logoUrl'] = wp_get_attachment_url($custom_logo_id);
        }

        // Localize the script with data
        wp_localize_script('richwp-seo-panel', 'richwpSeoData', $data);

        $show_notice = self::should_show_donation_notice();

        wp_localize_script('richwp-seo-panel', 'richwpDonationData', [
            'showNotice' => $show_notice,
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('richwp_seo_dismiss_donation')
        ]);
    }

    /**
     * Handle AJAX request for dismissing donation notice.
     */
    public static function dismiss_donation_notice() {
        check_ajax_referer('richwp_seo_dismiss_donation', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_die('Insufficient permissions');
        }

        $dismissal = get_option('richwp_seo_donation_dismissal', ['level' => 0, 'timestamp' => 0]);
        $dismissal['level'] = min($dismissal['level'] + 1, 4); // Max level 4
        $dismissal['timestamp'] = time();

        update_option('richwp_seo_donation_dismissal', $dismissal);

        wp_send_json_success();
    }

    /**
     * Show donation notice in admin.
     */
    public static function show_donation_notice() {
        $show_notice = self::should_show_donation_notice();

        if ($show_notice) {
            $nonce = wp_create_nonce('richwp_seo_dismiss_donation');
            ?>
            <div id="richwp-seo-donation-notice" class="notice notice-error is-dismissible">
                <p><strong><?php echo esc_html(__('☕ Support RichWP SEO Development', 'richwp-seo')); ?></strong></p>
                <p><?php echo esc_html(__('Enjoying this free plugin? Consider supporting ongoing development and new features.', 'richwp-seo')); ?></p>
                <p><a href="https://richwp.com/donate" target="_blank" rel="noopener noreferrer" class="button button-primary"><?php echo esc_html(__('Make a Donation', 'richwp-seo')); ?></a></p>
            </div>
            <script>
            jQuery(document).ready(function($) {
                $('#richwp-seo-donation-notice .notice-dismiss').on('click', function() {
                    $.post(ajaxurl, {
                        action: 'richwp_seo_dismiss_donation',
                        nonce: '<?php echo esc_js($nonce); ?>'
                    });
                });
            });
            </script>
            <?php
        }
    }
}