<?php
/**
 * Frontend Class for RichWP SEO
 *
 * Handles frontend output of SEO meta tags and schema.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class RichWP_SEO_Frontend {

    /**
     * Initialize the frontend handler.
     */
    public static function init() {
        add_filter('document_title_parts', [__CLASS__, 'custom_title'], 10, 1);
        add_action('wp_head', [__CLASS__, 'inject_meta'], 5);
    }

    /**
     * Override document title with custom title.
     *
     * @param array $title_parts The title parts.
     * @return array Modified title parts.
     */
    public static function custom_title($title_parts) {
        if (is_singular(['post', 'page'])) {
            global $post;
            $custom_title = get_post_meta($post->ID, '_richwp_custom_title', true);
            if ($custom_title) {
                return ['title' => $custom_title];
            }
        }
        return $title_parts;
    }

    /**
     * Inject meta description and schema JSON-LD into head.
     */
    public static function inject_meta() {
        if (!is_singular(['post', 'page'])) {
            return;
        }

        global $post;
        $meta_desc = get_post_meta($post->ID, '_richwp_meta_desc', true);
        $schema_json = get_post_meta($post->ID, '_richwp_schema_json', true);

        if ($meta_desc) {
            echo '<meta name="description" content="' . esc_attr($meta_desc) . '">' . "\n";
        }

        if ($schema_json && self::is_valid_json($schema_json)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo '<script type="application/ld+json">' . $schema_json . '</script>' . "\n";
        }
    }

    /**
     * Validate JSON string.
     *
     * @param string $json JSON string to validate.
     * @return bool True if valid JSON, false otherwise.
     */
    private static function is_valid_json($json) {
        if (!is_string($json) || empty($json)) {
            return false;
        }

        json_decode($json);
        return json_last_error() === JSON_ERROR_NONE;
    }
}