<?php
/**
 * Meta Handler Class for RichWP SEO
 *
 * Handles registration of custom meta fields for SEO data.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class RichWP_SEO_Meta_Handler {

    /**
     * Initialize the meta handler.
     */
    public static function init() {
        add_action('init', [__CLASS__, 'register_meta_fields']);
    }

    /**
     * Register custom meta fields for posts and pages.
     */
    public static function register_meta_fields() {
        $post_types = ['post', 'page'];

        foreach ($post_types as $post_type) {
            register_post_meta($post_type, '_richwp_custom_title', [
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'auth_callback' => '__return_true',
                'sanitize_callback' => 'sanitize_text_field'
            ]);

            register_post_meta($post_type, '_richwp_meta_desc', [
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'auth_callback' => '__return_true',
                'sanitize_callback' => 'sanitize_text_field'
            ]);

            register_post_meta($post_type, '_richwp_schema_json', [
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'auth_callback' => '__return_true',
                'sanitize_callback' => [__CLASS__, 'sanitize_json']
            ]);
        }
    }

    /**
     * Sanitize JSON input.
     *
     * @param mixed $value The value to sanitize.
     * @return string Sanitized JSON string or empty string.
     */
    public static function sanitize_json($value) {
        if (!is_string($value)) {
            return '';
        }

        $value = trim($value);

        if (empty($value)) {
            return '';
        }

        // Attempt to decode JSON
        $decoded = json_decode($value);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return '';
        }

        // Re-encode to ensure valid JSON structure, preserving formatting
        $re_encoded = json_encode($decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if ($re_encoded === false) {
            return '';
        }

        // Basic sanitization - remove script tags and other potentially dangerous content
        $sanitized = wp_strip_all_tags($re_encoded);
        $sanitized = preg_replace('/<script[^>]*>.*?<\/script>/is', '', $sanitized);

        // Final validation
        json_decode($sanitized);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return '';
        }

        return $sanitized;
    }
}